<?php

namespace App\Http\Controllers\Api;

use App\Enums\UserRole;
use App\Enums\VehicleLogType;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\VehicleLog\StoreVehicleLogRequest;
use App\Http\Requests\VehicleLog\UpdateVehicleLogRequest;
use App\Http\Resources\VehicleLogResource;
use App\Models\Vehicle;
use App\Models\VehicleLog;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\QueryBuilder\QueryBuilder;

class VehicleLogController extends Controller
{
    public function index()
    {
        $query = QueryBuilder::for(VehicleLog::class)
            ->allowedFilters(['vehicle_id', 'guard_id', 'log_type', 'logged_at'])
            ->allowedIncludes(['vehicle', 'guardUser'])
            ->latest('logged_at');

        $vehicleLogs = $query
            ->paginate(request("perPage", 15));

        return ApiResponse::paginated($vehicleLogs);
    }

    public function store(StoreVehicleLogRequest $request)
    {
        $vehicleLog = VehicleLog::create($request->validated());

        return ApiResponse::created(
            new VehicleLogResource($vehicleLog->load(['vehicle', 'guardUser'])),
            "Vehicle log has been created successfully!"
        );
    }

    public function show(VehicleLog $vehicleLog)
    {
        return ApiResponse::resource(
            new VehicleLogResource($vehicleLog->load(['vehicle', 'guardUser']))
        );
    }

    public function update(UpdateVehicleLogRequest $request, VehicleLog $vehicleLog)
    {
        $vehicleLog->update($request->validated());

        return ApiResponse::updated(
            new VehicleLogResource($vehicleLog->load(['vehicle', 'guardUser'])),
            "Vehicle log has been updated successfully!"
        );
    }

    public function destroy(VehicleLog $vehicleLog)
    {
        $vehicleLog->delete();

        return ApiResponse::deleted("Vehicle log has been deleted successfully!");
    }

    public function scan(Request $request)
    {
        $request->validate([
            'unique_code' => ['required', 'string'],
            'log_type' => ['required', Rule::enum(VehicleLogType::class)],
            'purpose' => ['nullable', 'string'],
            'outsider_vehicle_details' => ['nullable', 'array'],
        ]);

        if (auth()->user()->role !== UserRole::GUARD) {
            return ApiResponse::forbidden("You don't have permission to scan vehicle log!");
        }

        $vehicle = Vehicle::where('unique_code', $request->unique_code)
            ->where('is_active', true)
            ->first();

        if (!$vehicle) {
            return ApiResponse::notFound('Vehicle not found or inactive');
        }

        $vehicleLog = VehicleLog::create([
            'vehicle_id' => $vehicle->id,
            'guard_id' => auth()->user()->id,
            'log_type' => $request->log_type,
            'logged_at' => now(),
            'purpose' => $request->purpose,
            'outsider_vehicle_details' => $request->outsider_vehicle_details
        ]);

        return ApiResponse::created(
            new VehicleLogResource($vehicleLog->load(['vehicle', 'guardUser'])),
            "Vehicle " . $request->log_type . " logged successfully!"
        );
    }

    public function today()
    {
        $today = now()->format('Y-m-d');

        $query = QueryBuilder::for(VehicleLog::class)
            ->whereDate('logged_at', $today)
            ->allowedFilters(['vehicle_id', 'guard_id', 'log_type'])
            ->allowedIncludes(['vehicle', 'guardUser'])
            ->latest('logged_at');

        $vehicleLogs = $query
            ->paginate(request("perPage", 15));

        return ApiResponse::paginated($vehicleLogs);
    }
}
