<?php

namespace App\Http\Controllers\Api;

use App\Enums\UserRole;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\DailyPatrolSummary\StoreDailyPatrolSummaryRequest;
use App\Http\Requests\DailyPatrolSummary\UpdateDailyPatrolSummaryRequest;
use App\Http\Resources\DailyPatrolSummaryResource;
use App\Models\DailyPatrolSummary;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Spatie\QueryBuilder\QueryBuilder;
use Spatie\QueryBuilder\AllowedFilter;

class DailyPatrolSummaryController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = QueryBuilder::for(DailyPatrolSummary::class)
            ->allowedFilters([
                'guard_id',
                'patrol_round_id',
                'patrol_date',
                'status',
                AllowedFilter::scope('today'),
                AllowedFilter::scope('this_week'),
                AllowedFilter::scope('this_month')
            ])
            ->allowedIncludes(['guardUser', 'patrolRound', 'patrollingLogs'])
            ->allowedSorts(['patrol_date', 'actual_start_time', 'completion_percentage', 'created_at'])
            ->defaultSort('-patrol_date', '-actual_start_time');

        $user = auth()->user();

        if ($user->role == UserRole::GUARD) {
            $query->where('guard_id', $user->id);
        }

        $summaries = $query->paginate($request->get('per_page', 15));

        return ApiResponse::paginated($summaries, 'Daily patrol summaries retrieved successfully');
    }

    public function store(StoreDailyPatrolSummaryRequest $request): JsonResponse
    {
        $summary = DailyPatrolSummary::create($request->validated());

        return ApiResponse::created(
            new DailyPatrolSummaryResource($summary->load(['guardUser', 'patrolRound'])),
            'Daily patrol summary created successfully'
        );
    }

    public function show(DailyPatrolSummary $dailyPatrolSummary): JsonResponse
    {
        $summary = $dailyPatrolSummary->load(['guardUser', 'patrolRound', 'patrollingLogs.location']);

        return ApiResponse::success(
            new DailyPatrolSummaryResource($summary),
            'Daily patrol summary retrieved successfully'
        );
    }

    public function update(UpdateDailyPatrolSummaryRequest $request, DailyPatrolSummary $dailyPatrolSummary): JsonResponse
    {
        $dailyPatrolSummary->update($request->validated());

        return ApiResponse::success(
            new DailyPatrolSummaryResource($dailyPatrolSummary->load(['guardUser', 'patrolRound'])),
            'Daily patrol summary updated successfully'
        );
    }

    public function destroy(DailyPatrolSummary $dailyPatrolSummary): JsonResponse
    {
        // Check if there are associated patrolling logs
        if ($dailyPatrolSummary->patrollingLogs()->exists()) {
            return ApiResponse::error(
                'Cannot delete patrol summary that has associated patrol logs',
                422
            );
        }

        $dailyPatrolSummary->delete();

        return ApiResponse::success(
            null,
            'Daily patrol summary deleted successfully'
        );
    }

    public function today(): JsonResponse
    {
        $today = now()->format('Y-m-d');
        $user = auth()->user();

        $query = QueryBuilder::for(DailyPatrolSummary::class)
            ->where('patrol_date', $today);

        if ($user->role == UserRole::GUARD) {
            $query->where('guard_id', $user->id);
        }

        $summaries = $query->paginate(request('per_page', 15));

        return ApiResponse::paginated($summaries, 'Today\'s patrol summaries retrieved successfully');
    }
}
