<?php

namespace App\Http\Controllers\Api;

use App\Enums\DailyPatrolSummaryStatus;
use App\Enums\PatrollingStatus;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\PatrollingLog\StorePatrollingLogRequest;
use App\Http\Requests\PatrollingLog\UpdatePatrollingLogRequest;
use App\Http\Resources\PatrollingLogResource;
use App\Models\DailyPatrolSummary;
use App\Models\PatrollingLocation;
use App\Models\PatrollingLog;
use App\Models\PatrolRound;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class PatrollingLogController extends Controller
{
    public function index()
    {
        $query = QueryBuilder::for(PatrollingLog::class)
            ->allowedFilters([
                'location_id',
                'guard_id',
                'patrol_date',
                'status',
                AllowedFilter::scope('today'),
                AllowedFilter::scope('this_week'),
                AllowedFilter::scope('this_month')
            ])
            ->allowedIncludes(['location', 'guardUser'])
            ->latest();

        $logs = $query
            ->paginate(request("perPage", 15));

        return ApiResponse::paginated($logs);
    }

    public function store(StorePatrollingLogRequest $request)
    {
        $log = PatrollingLog::create($request->validated());

        return ApiResponse::created(
            new PatrollingLogResource($log->load(['location', 'guardUser'])),
            "Patrolling log has been created successfully!"
        );
    }

    public function show(PatrollingLog $patrollingLog)
    {
        return ApiResponse::resource(
            new PatrollingLogResource($patrollingLog->load(['location', 'guardUser']))
        );
    }

    public function update(UpdatePatrollingLogRequest $request, PatrollingLog $patrollingLog)
    {
        $patrollingLog->update($request->validated());

        return ApiResponse::updated(
            new PatrollingLogResource($patrollingLog->load(['location', 'guardUser'])),
            "Patrolling log has been updated successfully!"
        );
    }

    public function destroy(PatrollingLog $patrollingLog)
    {
        $patrollingLog->delete();

        return ApiResponse::deleted("Patrolling log has been deleted successfully!");
    }

    public function scan(Request $request)
    {
        $request->validate([
            'location_code' => ['required', 'string'],
            'daily_patrol_summary_id' => ['required', 'exists:daily_patrol_summaries,id'],
            'status' => ['required', Rule::enum(PatrollingStatus::class)],
            'notes' => ['nullable', 'string'],
            'issues_found' => ['nullable', 'string']
        ]);

        $location = PatrollingLocation::where('location_code', $request->location_code)
            ->where('is_active', true)
            ->first();

        if (!$location) {
            return ApiResponse::notFound('Location not found or inactive');
        }

        // Verify the daily patrol summary belongs to the authenticated guard
        $summary = DailyPatrolSummary::where('id', $request->daily_patrol_summary_id)
            ->where('guard_id', auth()->user()->id)
            ->where('status', DailyPatrolSummaryStatus::STARTED)
            ->first();

        if (!$summary) {
            return ApiResponse::notFound('Active patrol round not found');
        }

        $log = PatrollingLog::create([
            'location_id' => $location->id,
            'guard_id' => auth()->user()->id,
            'daily_patrol_summary_id' => $request->daily_patrol_summary_id,
            'patrol_date' => now()->format('Y-m-d'),
            'patrol_time' => now(),
            'status' => $request->status,
            'notes' => $request->notes,
            'issues_found' => $request->issues_found
        ]);

        return ApiResponse::created(
            new PatrollingLogResource($log->load(['location', 'guardUser', 'dailyPatrolSummary'])),
            "Location patrolled successfully!"
        );
    }

    public function startPatrolRound(Request $request)
    {
        $request->validate([
            'patrol_round_id' => ['required', 'exists:patrol_rounds,id']
        ]);

        $today = now()->format('Y-m-d');
        $guardId = auth()->user()->id;

        // Check if guard already has an active round for this patrol round today
        $existingSummary = DailyPatrolSummary::where('guard_id', $guardId)
            ->where('patrol_date', $today)
            ->where('patrol_round_id', $request->patrol_round_id)
            // ->where('status', DailyPatrolSummaryStatus::STARTED)
            ->first();

        if ($existingSummary) {
            return ApiResponse::error('You already have an active patrol round for this time slot', 422);
        }

        $patrolRound = PatrolRound::findOrFail($request->patrol_round_id);

        if (!$patrolRound->is_active) {
            return ApiResponse::error('This patrol round is not active', 422);
        }

        $summary = DailyPatrolSummary::create([
            'guard_id' => $guardId,
            'patrol_round_id' => $request->patrol_round_id,
            'patrol_date' => $today,
            'total_locations' => PatrollingLocation::where('is_active', true)->count(),
            'completed_locations' => 0,
            'missed_locations' => 0,
            'expected_start_time' => $patrolRound->start_time,
            'expected_end_time' => $patrolRound->end_time,
            'actual_start_time' => now(),
            'status' => DailyPatrolSummaryStatus::STARTED,
        ]);

        return ApiResponse::success(
            $summary->load('patrolRound'),
            "Patrol round started successfully!"
        );
    }

    public function completePatrolRound(Request $request)
    {
        $request->validate([
            'daily_patrol_summary_id' => ['required', 'exists:daily_patrol_summaries,id'],
            'notes' => ['nullable', 'string']
        ]);

        $guardId = auth()->user()->id;

        $summary = DailyPatrolSummary::where('id', $request->daily_patrol_summary_id)
            ->where('guard_id', $guardId)
            ->where('status', DailyPatrolSummaryStatus::STARTED)
            ->first();

        if (!$summary) {
            return ApiResponse::notFound("No active patrol round found");
        }

        $completedLocations = PatrollingLog::where('daily_patrol_summary_id', $request->daily_patrol_summary_id)
            ->count();

        $summary->update([
            'completed_locations' => $completedLocations,
            'missed_locations' => $summary->total_locations - $completedLocations,
            'actual_end_time' => now(),
            'status' => DailyPatrolSummaryStatus::COMPLETED,
            'notes' => $request->notes
        ]);

        return ApiResponse::success(
            $summary->load('patrolRound'),
            "Patrol round completed successfully!"
        );
    }
}
